import statsapi, sys
import pandas as pd
from pandas import DataFrame
from datetime import datetime, timedelta
import json
import game_selection

def get_lineups(gameID):
	game_timestamps=statsapi.get('game_timestamps',{'gamePk':gameID})
	if game_timestamps==[]:
		sys.exit("\033[1mBoth lineups have not yet been posted.\033[0m")
	game_data=statsapi.boxscore_data(gameID,timecode=game_timestamps[0])
	away_batters=DataFrame(game_data['awayBatters'])
	home_batters=DataFrame(game_data['homeBatters'])
	away_pitchers=DataFrame(game_data['awayPitchers'])
	home_pitchers=DataFrame(game_data['homePitchers'])
	
	away_batters=away_batters[(away_batters["substitution"]==False) & (away_batters["battingOrder"]!="")]
	home_batters=home_batters[(home_batters["substitution"]==False) & (home_batters["battingOrder"]!="")]
	
	away_pitchers=away_pitchers[away_pitchers['personId']!=0]
	home_pitchers=home_pitchers[home_pitchers['personId']!=0]
	
	return away_batters, home_batters, away_pitchers, home_pitchers
	#NB: batters (and presumably pitchers) data has stats that could be included
	
def parse_roster_entry(entry):
	player_data={
		'id': entry['person']['id'],
		'jerseyNumber': entry['jerseyNumber'],
		'position_type': entry['position']['type'],
		#'IntNum': int(entry['jerseyNumber'])
	}
	return player_data
	
def get_rosters(game,date):
	away_team_roster=statsapi.get('team_roster',{'teamId': game['Away Team ID'],'date': date})
	home_team_roster=statsapi.get('team_roster',{'teamId': game['Home Team ID'],'date': date})
	
	away_team_roster=DataFrame([parse_roster_entry(entry) for entry in away_team_roster['roster']])
	home_team_roster=DataFrame([parse_roster_entry(entry) for entry in home_team_roster['roster']])
	
	return away_team_roster, home_team_roster

def numeric_positions():
	posnums=DataFrame({
		"position": ["DH","P","C","1B","2B","3B","SS","LF","CF","RF"],
		"posnum": [str(i) for i in range(10)]
		})
	return posnums
	
def merge_rosters(away_batters, home_batters, away_pitchers, home_pitchers, away_team_roster, home_team_roster):
	away_batters = away_batters.merge(away_team_roster, how='left', validate='one_to_one', left_on='personId', right_on='id')
	home_batters = home_batters.merge(home_team_roster, how='left', validate='one_to_one', left_on='personId', right_on='id')

	posnums = numeric_positions()
	away_batters = away_batters.merge(posnums, how='left', validate='one_to_one', on='position')
	away_batters.sort_values("battingOrder", inplace=True)
	home_batters = home_batters.merge(posnums, how='left', validate='one_to_one', on='position')
	home_batters.sort_values("battingOrder", inplace=True)

	away_pitchers = away_pitchers.merge(away_team_roster, how='left', validate='one_to_one', left_on='personId', right_on='id')
	home_pitchers = home_pitchers.merge(home_team_roster, how='left', validate='one_to_one', left_on='personId', right_on='id')
	
	return away_batters, home_batters, away_pitchers, home_pitchers
	
def parse_detail(player):
	player_data={
		'id': player['id'],
		'Bats': player['batSide']['code'],
		'Name': player['boxscoreName'],
		'Throws': player['pitchHand']['code'],
		'Group': player['primaryPosition']['type'],
		'PrimPos': player['primaryPosition']['abbreviation']
	}
	try:
		for entry in player['stats']:
			is_pitching=0
			if entry['group']['displayName']=="pitching":
				is_pitching=1
			for split in entry['splits']:
				if split['sport']['id']==0:
					if is_pitching==1:
						stats = {key + "_p": value for key, value in split['stat'].items()}
					else:
						stats = split['stat']
					player_data=player_data | stats
	except:
		pass
	
	return player_data
	
def stats_correction(date):
	print("1. Use the default date: "+date)
	print("2. Specify a date (will include current season stats only)")
	while True:
		selection=input(" > ").strip()
		
		if selection.isdigit():
			choice=int(selection)
			if choice in [1,2]:
				return choice
			else:
				print("Invalid input. Try again")
				
def set_stats_parameters(game,sportID,date):
	date = (
		    datetime.strptime(date, "%Y-%m-%d")
    		- timedelta(days=1)
			).strftime("%Y-%m-%d")
	season_info = statsapi.get('season',{"seasonId": game['season'], "sportId":sportID})
	if game['gameType'] in ['R','A']:
		start_date=season_info['seasons'][0]['regularSeasonStartDate']
		game_type_str="R"
	elif game['gameType'] in ['P','F','D','L','W']:
		start_date=season_info['seasons'][0]['regularSeasonEndDate'] 
		start_date = (
		    datetime.strptime(start_date, "%Y-%m-%d")
    		+ timedelta(days=1)
			).strftime("%Y-%m-%d")
		#game_type_str="[P,F,D,L,W]"
		game_type_str="P"
	elif game['gameType'] in ['S']:
		start_date = season_info['seasons'][0]['preSeasonStartDate']
		game_type_str="S"
	if start_date>date: #Make it so this option arises only once
		print("")
		print("Default start date for stats is after the game. Do you want to use the default correction (start of the regular season) or specify a start date for the stats?")
		choice=stats_correction(season_info['seasons'][0]['regularSeasonStartDate'])
		if choice==1:
			start_date=season_info['seasons'][0]['regularSeasonStartDate']
		if choice==2:
			print("")
			print("Select a date")
			start_date=game_selection.get_date()
		game_type_str="[P,R]"
	
	return start_date, date, game_type_str, game['season']
				
def get_player_details(roster,game,sportID,start_date,end_date,game_type_str,season):
	player_ids = roster['id'].tolist()
	details=statsapi.get('people',{
		"personIds": ",".join(map(str, player_ids)),
		"hydrate": (
            "stats("
            "group=[hitting,pitching],"
            "type=[byDateRange],"
            f"season={season},"
            f"startDate={start_date},"
            f"endDate={end_date},"
            f"sportId={sportID},"
            f"gameType={game_type_str},"
            ")"
        ),
	}
	)
	
	player_details=DataFrame([parse_detail(x) for x in details['people']])
	
	if 'inningsPitched_p' in player_details.columns:
		player_details["inningsPitched_p"] = pd.to_numeric(player_details["inningsPitched_p"], errors="coerce").fillna('')
	
	updated_roster=roster.merge(player_details,how='outer',validate='one_to_one',on='id')
	
	updated_roster=updated_roster.fillna("")
	
	return updated_roster
	
	#Use sport-code = "All"
	#Check for both pitching and hitting stats