#!/usr/bin/env python3

'''
Scorecard Generator v2.0, 1/10/26

Copyright (c) 2026 Ariell Zimran

This software is released under the MIT License.
See: [https://opensource.org/licenses/MIT](https://opensource.org/licenses/MIT)

This application and its author are not affiliated with Major League Baseball or any MLB team.
Use of MLB data is subject to the notice at: http://gdx.mlb.com/components/copyright.txt

To compile:
	python3 -m PyInstaller --onefile ScorecardGenerator.py
	or
	python3 -m PyInstaller --onedir --noupx ScorecardGenerator.py
	to not have it take forever to start up
	
Major changes from versions 1.X:
	Restructure to use functions to simplify everything
	Add detailed statistics
	Add coverage of non-MLB leagues
	
Future goals:
	Handle Cactus and Grapefruit Leagues, Arizona Fall League
	Time zone support
	Special event logos (playoffs)
	Handle postponements
		
Logic of the program:
	1. Open by loading the current date's MLB games, or indicating that there are none.
		a. Present the user with the option to choose a numbered game or a lettered option
			i. Options: toggle manual mode, change date, change league
		b. Return selected game information (teams, time, gameId)
	2. Collect starting lineup and put into dataframe
	3. Collect rosters and put into dataframe
	4. Merge starting lineup and rosters to get all necesssary info (esp. jersey number)
	5. Output table in manual mode or card in regular mode
'''

import warnings, colorama
import pandas as pd
from time import sleep
warnings.filterwarnings("ignore")
from colorama import init, Style
init()

def bold(text):
	print(Style.BRIGHT+text+Style.RESET_ALL)
	
version_text="v2.0"
bold("Scorecard Generator "+version_text)
bold("Created by Ariell Zimran")
sleep(1)
import sys, os, statsapi, re, pycurl, json, io, prettytable
from prettytable import PrettyTable
import game_selection, get_data, display, graphics_packages

#Select game to score and mode
game, manual_mode, year, month, day, month_str, sportID, full_stats, records, show_logos = game_selection.select_game()
if records==0:
	game['away_record']['wins']='X'
	game['away_record']['losses']='X'
	game['home_record']['wins']='X'
	game['home_record']['losses']='X'
#print(game)
#stop=input("")

#Get starting lineups
print("")
print("Getting Lineups")
away_batters, home_batters, away_pitchers, home_pitchers = get_data.get_lineups(game['ID'])

#Get team rosters; quick version for manual mode, longer version for full mode
print("")
print("Getting Rosters")
away_team_roster, home_team_roster = get_data.get_rosters(game,year+"-"+month+"-"+day)

if manual_mode==0:
	print("")
	start_date, end_date, game_type_str, season = get_data.set_stats_parameters(game,sportID,year+"-"+month+"-"+day)
	print("")
	print("Getting Detailed Roster Data")
	away_team_roster = get_data.get_player_details(away_team_roster,game,sportID,start_date,end_date,game_type_str,season)
	home_team_roster = get_data.get_player_details(home_team_roster,game,sportID,start_date,end_date,game_type_str,season)

#Merge roster info
away_batters, home_batters, away_pitchers, home_pitchers = get_data.merge_rosters(away_batters, home_batters, away_pitchers, home_pitchers, away_team_roster, home_team_roster)

#Display results
if manual_mode==1:
	display.manual_tables(away_batters, away_pitchers, home_batters, home_pitchers)
	
if manual_mode==0:
	
	if show_logos==1:
		try:
			if sportID==1:
				graphics_pack=graphics_packages.mlb_package(game)
			elif sportID==11:
				graphics_pack=graphics_packages.aaa_package(game)
			elif sportID==12:
				graphics_pack=graphics_packages.aa_package(game)
			elif sportID==13:
				graphics_pack=graphics_packages.higha_package(game)
			elif sportID==14:
				graphics_pack=graphics_packages.lowa_package(game)
			elif sportID==17:
				graphics_pack=graphics_packages.winter_package(game)
			elif sportID==23:
				graphics_pack=graphics_packages.lmb_package(game)
			elif sportID==51:
				graphics_pack=graphics_packages.wbc_package(game)
		except KeyError:
			print("")
			print("Missing some logos. Switching to no-logo mode.")
			show_logos=0
			
		
	tex_contents=display.create_latex(away_batters, home_batters, away_pitchers, home_pitchers, away_team_roster, home_team_roster, game, sportID, month_str, day, year, full_stats, show_logos)
	#print(tex_contents)
		
	#with open("Test.tex", "w") as f:
	#   f.write(tex_contents)
	
	print("")
	
	save_file=input("\033[1mWhat should I save the scorecard as?\033[0m\n > ")
	
	print("")
	
	print("Compiling Scorecard")
	
	#Prepare data to send to the LaTeX compiler
	payload = {
		"resources": [
			{
				"main": True,
				"content": tex_contents
			}
		]
	}
	
	if show_logos==1:
		payload['resources']+=graphics_pack
	
	# Convert payload to JSON bytes
	json_data = json.dumps(payload).encode('utf-8')
	
	# Create a buffer to hold the PDF data
	pdf_buffer = io.BytesIO()
	hdr_buffer = io.BytesIO()
	
	# Send the POST request and write response directly to buffer
	c = pycurl.Curl()
	c.setopt(c.URL, "https://latex.ytotech.com/builds/sync")
	c.setopt(c.HTTPHEADER, ["Content-Type: application/json"])
	c.setopt(c.POSTFIELDS, json_data)
	c.setopt(c.WRITEDATA, pdf_buffer)
	c.setopt(c.HEADERFUNCTION, hdr_buffer.write)
	c.perform()
	status_code = c.getinfo(pycurl.RESPONSE_CODE)
	content_type = c.getinfo(pycurl.CONTENT_TYPE)
	c.close()
	
	is_pdf = (
    	status_code == 201 and
    	content_type is not None and
	    "application/pdf" in content_type.lower()
	)
	
	if not is_pdf:
		body_bytes = pdf_buffer.getvalue()
		body_text = body_bytes.decode("utf-8", errors="replace")
		
		print("Compilation failed. Details below. If the error comes from the images, consider retrying with logos toggled off.")
		print("HTTP status:", status_code)
		print("Content-Type:", content_type)
		print("----- Response headers -----")
		print(hdr_buffer.getvalue().decode("iso-8859-1", errors="replace"))
		print("----- Compilation log / error payload -----")
		print(body_text)
	else:
		#Determine if the code is being run from an executable and change directory if necessary
		if getattr(sys, 'frozen', False):
			output_dir=os.path.dirname(sys.executable)
		else:
			output_dir="."
	
		# Save PDF data to file
		with open(output_dir+"/"+save_file+".pdf", "wb") as f:
			f.write(pdf_buffer.getvalue())
	
		print("")
		bold("PDF successfully saved as "+save_file+".pdf")
	
		print("")
		print("Logos from sportslogos.net, Wikipedia, or team websites")
		print("LaTeX compilation via YtoTech")
	
print("")

print("Done")