
"""
core.py - server-side PDF generator for ScorecardGenerator

This module refactors your CLI flow into callable functions.
It reuses your existing modules: game_selection, get_data, display, graphics_packages.

Notes:
- manual_mode is intentionally not supported in PDF generation (it was CLI table-only).
- stats_date_mode:
    - "default": use game date (same behavior as original)
    - "specify": use provided stats_date (YYYY-MM-DD) for season-to-date stats only
"""

from __future__ import annotations

import io
import json
from typing import Any, Dict, Optional, Tuple

import statsapi

import game_selection
import get_data
import display
import graphics_packages, math

from datetime import date as _date

SPORTS = {
    1: {"id": 1, "name": "MLB"},
    11: {"id": 11, "name": "AAA"},
    12: {"id": 12, "name": "AA"},
    13: {"id": 13, "name": "High-A"},
    14: {"id": 14, "name": "Low-A"},
    16: {"id": 16, "name": "Rookie"},
    17: {"id": 17, "name": "Winter Leagues"},
    23: {"id": 23, "name": "LMB"},
    51: {"id": 51, "name": "WBC"},
}

def sanitize_json(obj):
    if isinstance(obj, float):
        if math.isnan(obj) or math.isinf(obj):
            return None
        return obj
    if isinstance(obj, dict):
        return {k: sanitize_json(v) for k, v in obj.items()}
    if isinstance(obj, list):
        return [sanitize_json(v) for v in obj]
    return obj


def list_schedule(date: str, sport_id: int) -> Dict[str, Any]:
    """
    Returns a schedule payload with a list of games for the date + sport_id.
    Each game includes the fields used by your generator.
    """
    schedule = statsapi.get("schedule", {"sportId": sport_id, "date": date})
    games, year, month, day, month_str, totalGames = game_selection.process_schedule(schedule)

    out_games = []
    for g in games.values():
        out_games.append({
            "id": int(g["ID"]),
            "away": g["Away Team"],
            "home": g["Home Team"],
            "gameDateUTC": g["gameDateUTC"],
            "venue": g.get("Venue", ""),
            "awayRecord": f"{g['away_record']['wins']}-{g['away_record']['losses']}",
            "homeRecord": f"{g['home_record']['wins']}-{g['home_record']['losses']}",
        })

    return sanitize_json({
        "date": date,
        "sportId": sport_id,
        "sportName": SPORTS.get(sport_id, {}).get("name", str(sport_id)),
        "year": year,
        "month": month,
        "day": day,
        "monthStr": month_str,
        "games": out_games,
        "totalGames": totalGames
    })


def _find_game_in_schedule(game_id: int, date: str, sport_id: int) -> Tuple[dict, str, str, str, str]:
    schedule = statsapi.get("schedule", {"sportId": sport_id, "date": date})
    games, year, month, day, month_str, totalGames = game_selection.process_schedule(schedule)

    for g in games.values():
        if int(g["ID"]) == int(game_id):
            return g, year, month, day, month_str

    raise ValueError("Game not found in schedule for that date/league. Try refreshing the schedule.")


def _graphics_pack_for_sport(game: dict, sport_id: int) -> list:
    if sport_id == 1:
        return graphics_packages.mlb_package(game)
    if sport_id == 11:
        return graphics_packages.aaa_package(game)
    if sport_id == 12:
        return graphics_packages.aa_package(game)
    if sport_id == 13:
        return graphics_packages.higha_package(game)
    if sport_id == 14:
        return graphics_packages.lowa_package(game)
    if sport_id == 17:
        return graphics_packages.winter_package(game)
    if sport_id == 23:
        return graphics_packages.lmb_package(game)
    if sport_id == 51:
        return graphics_packages.wbc_package(game)
    return []


def compile_latex_to_pdf(tex_contents: str, resources: Optional[list] = None) -> bytes:
    """
    Uses the same YtoTech LaTeX endpoint you used in CLI.
    Returns PDF bytes or raises RuntimeError with details.
    """
    payload = {
        "resources": [
            {"main": True, "content": tex_contents}
        ]
    }
    if resources:
        payload["resources"] += resources

    json_data = json.dumps(payload).encode("utf-8")

    # Prefer requests if available; fall back to urllib.
    try:
        import requests  # type: ignore
        resp = requests.post(
            "https://latex.ytotech.com/builds/sync",
            data=json_data,
            headers={"Content-Type": "application/json"},
            timeout=60,
        )
        ct = resp.headers.get("Content-Type", "")
        # YtoTech may return 201 (Created) with a PDF body.
        is_pdf = ("application/pdf" in ct) or resp.content[:4] == b"%PDF"
        if resp.status_code not in (200, 201) or not is_pdf:
            preview = resp.text[:400] if hasattr(resp, "text") else str(resp.content[:200])
            raise RuntimeError(f"LaTeX build failed: HTTP {resp.status_code} {preview}")
        return resp.content
    except ImportError:
        import urllib.request
        req = urllib.request.Request(
            "https://latex.ytotech.com/builds/sync",
            data=json_data,
            headers={"Content-Type": "application/json"},
            method="POST",
        )
        with urllib.request.urlopen(req, timeout=60) as resp:
            ct = resp.headers.get("Content-Type", "")
            data = resp.read()
            is_pdf = ("application/pdf" in ct) or data[:4] == b"%PDF"
            if not is_pdf:
                raise RuntimeError(f"LaTeX build failed: Content-Type={ct} body={data[:200]}")
            return data

import re

_ALLOWED = {"S", "R", "P", "F", "D", "L", "W", "E"}
_POST_SUBS = {"F", "D", "L", "W"}
_ORDER = ["E", "S", "R", "P", "F", "D", "L", "W"]

def normalize_stats_game_type(gt: str) -> str:
    gt = gt.strip()

    # Single code
    if gt in _ALLOWED:
        return gt

    # Bracket list like [S,R,P] or [F,D]
    m = re.fullmatch(r"\[([ESRPFDLW](,[ESRPFDLW])*)\]", gt)
    if not m:
        raise ValueError(f"Invalid statsGameType: {gt}")

    parts = [p.strip() for p in gt[1:-1].split(",") if p.strip()]
    parts_set = set(parts)

    if not parts_set.issubset(_ALLOWED):
        raise ValueError(f"Invalid statsGameType: {gt}")

    # Avoid double counting: if P is present, drop any postseason subsets
    if "P" in parts_set:
        parts_set -= _POST_SUBS

    # If all 4 postseason subsets are present (and P is not), normalize to P
    if _POST_SUBS.issubset(parts_set):
        parts_set -= _POST_SUBS
        parts_set.add("P")

    # If only one remains, return it directly
    if len(parts_set) == 1:
        return next(iter(parts_set))

    # Stable ordering for consistent output
    normalized = [c for c in _ORDER if c in parts_set]
    return "[" + ",".join(normalized) + "]"

def generate_pdf_bytes(
    *,
    game_id: int,
    date: str,         # YYYY-MM-DD
    sport_id: int,
    full_stats: bool,
    show_logos: bool,
    stats_date_mode: str = "default",  # "default" or "specify"
    stats_date: Optional[str] = None,  # YYYY-MM-DD if specify
    stats_start_date: Optional[str] = None,
    stats_game_type: Optional[str] = None,
    blank_if_no_lineups: bool = False,
) -> bytes:
    """
    Main entry point: generate PDF bytes based on wizard selections.
    """
    game, year, month, day, month_str = _find_game_in_schedule(game_id, date, sport_id)

    # Lineups + rosters
    lineup_missing = False
    try:
        away_batters, home_batters, away_pitchers, home_pitchers = get_data.get_lineups(game["ID"])
    except SystemExit as e:
        if not blank_if_no_lineups:
            raise
        lineup_missing = True
        away_batters = None
        home_batters = None
        away_pitchers = None
        home_pitchers = None

    roster_stats_date = _date.today().isoformat() if lineup_missing else f"{year}-{month}-{day}"

    away_team_roster, home_team_roster = get_data.get_rosters(game, roster_stats_date)

    # Detailed roster data (only when full mode)
    start_date, end_date, game_type_str, season = safe_set_stats_parameters(
        game=game,
        sport_id=sport_id,
        date=roster_stats_date if lineup_missing else date,
        stats_date_mode=stats_date_mode,
        stats_start_date=stats_date
    )
    
    if stats_game_type is not None:
        game_type_str = normalize_stats_game_type(stats_game_type)

    if stats_start_date is not None:
        start_date = stats_start_date

    away_team_roster = get_data.get_player_details(
        away_team_roster, game, sport_id, start_date, end_date, game_type_str, season
    )
    home_team_roster = get_data.get_player_details(
        home_team_roster, game, sport_id, start_date, end_date, game_type_str, season
    )

    # Merge roster info
    away_batters, home_batters, away_pitchers, home_pitchers = get_data.merge_rosters(
        away_batters, home_batters, away_pitchers, home_pitchers, away_team_roster, home_team_roster, lineup_missing
    )

    # Logos / graphics pack
    resources = None
    if show_logos:
        try:
            resources = _graphics_pack_for_sport(game, sport_id)
        except KeyError:
            # fall back to no logos
            show_logos = False
            resources = None

    # Generate LaTeX
    tex_contents = display.create_latex(
        away_batters, home_batters,
        away_pitchers, home_pitchers,
        away_team_roster, home_team_roster,
        game, sport_id, month_str, day, year,
        1 if full_stats else 0,
        1 if show_logos else 0,
        start_date if full_stats else None,
        end_date if full_stats else None,
        game_type_str if full_stats else None
    )

    # Compile to PDF
    return compile_latex_to_pdf(tex_contents, resources)


def manual_preview_text(*, game_id: int, date: str, sport_id: int) -> str:
    """
    Web wrapper for your CLI 'manual mode'.

    IMPORTANT: This does NOT modify any of your underlying modules.
    It simply calls the same prep steps your CLI does before printing tables:
      - get_lineups()
      - get_rosters()
      - merge_rosters()
      - display.manual_tables(...)
    """
    game, year, month, day, month_str = _find_game_in_schedule(game_id, date, sport_id)

    import io
    from contextlib import redirect_stdout

    try:
        away_batters, home_batters, away_pitchers, home_pitchers = get_data.get_lineups(game["ID"])
    except SystemExit as e:
        # get_lineups() uses sys.exit when lineups aren't posted yet
        return str(e)

    away_team_roster, home_team_roster = get_data.get_rosters(game, f"{year}-{month}-{day}")

    away_batters, home_batters, away_pitchers, home_pitchers = get_data.merge_rosters(
        away_batters, home_batters, away_pitchers, home_pitchers, away_team_roster, home_team_roster
    )

    buf = io.StringIO()
    with redirect_stdout(buf):
        display.manual_tables(away_batters, away_pitchers, home_batters, home_pitchers)

    return buf.getvalue()


def stats_correction_needed(*, game: dict, sport_id: int, date: str) -> dict:
    """
    Returns whether get_data.set_stats_parameters() would ask the interactive
    'default start date after game' question for this game/date.

    This does not modify any underlying code; it's only for the web UI.
    """
    from datetime import datetime, timedelta
    date_adj = (datetime.strptime(date, "%Y-%m-%d") - timedelta(days=1)).strftime("%Y-%m-%d")

    season_info=get_data.get_season_info(game,sport_id,date)
    #print(json.dumps(season_info,indent=4))
    game_type = game.get('gameType')

    start_date = None
    if game_type in ['R','A']:
        start_date = season_info['regularSeasonStartDate']
    elif game_type in ['P','F','D','L','W']:
        start_date = season_info['postSeasonStartDate']
    elif game_type in ['S','E']:
        start_date = season_info['preSeasonStartDate']

    needs = (start_date is not None) and (start_date > date)
    return {
        "needsCorrection": bool(needs),
        "defaultStartDate": season_info.get('regularSeasonStartDate'),
        "computedStartDate": start_date,
        "dateUsed": date_adj,
        "gameType": game_type,
        "seasonInfo": {
            "preseasonStart": season_info.get("preSeasonStartDate"),
            "regularSeasonStart": season_info.get("regularSeasonStartDate"),
            "postseasonStart": season_info.get("postSeasonStartDate"),
        },
    }


def safe_set_stats_parameters(
    *,
    game: dict,
    sport_id: int,
    date: str,
    stats_date_mode: str,
    stats_start_date: str | None,
):
    """
    Wrapper around get_data.set_stats_parameters that prevents any stdin prompts.

    If the underlying function would ask the interactive question, we answer it
    using stats_date_mode + stats_start_date, without changing your underlying modules.
    """
    info = stats_correction_needed(game=game, sport_id=sport_id, date=date)

    if not info["needsCorrection"]:
        # Safe: no interactive prompts will occur.
        return get_data.set_stats_parameters(game, sport_id, date)

    # If correction is needed, emulate the original prompt choice:
    # 1) use regularSeasonStartDate, or 2) specify a start date
    if stats_date_mode not in ("default", "specify"):
        stats_date_mode = "default"

    if stats_date_mode == "specify" and not stats_start_date:
        raise ValueError("This game requires a stats start-date choice. Choose 'Specify stats date' and pick a date.")

    orig_stats_correction = getattr(get_data, "stats_correction", None)
    orig_get_date = getattr(game_selection, "get_date", None)

    try:
        def _fake_stats_correction(_date):
            return 1 if stats_date_mode == "default" else 2

        def _fake_get_date():
            return stats_start_date

        if orig_stats_correction:
            get_data.stats_correction = _fake_stats_correction  # type: ignore
        if orig_get_date:
            game_selection.get_date = _fake_get_date  # type: ignore

        return get_data.set_stats_parameters(game, sport_id, date)
    finally:
        if orig_stats_correction:
            get_data.stats_correction = orig_stats_correction  # type: ignore
        if orig_get_date:
            game_selection.get_date = orig_get_date  # type: ignore
            
def stats_defaults(*, game_id: int, date: str, sport_id: int) -> dict:
    """
    Returns the same startDate/gameType defaults the app would use if the user did nothing.
    This uses safe_set_stats_parameters(...) so it matches current behavior.
    """
    game, _, _, _, _ = _find_game_in_schedule(game_id, date, sport_id)

    start_date, end_date, game_type_str, season = safe_set_stats_parameters(
        game=game,
        sport_id=sport_id,
        date=date,
        stats_date_mode="default",
        stats_start_date=None,
    )

    return {
        "startDate": start_date,
        "endDate": end_date,
        "gameType": game_type_str,
        "season": season,
    }

