
from __future__ import annotations

from fastapi import FastAPI, HTTPException
from fastapi.responses import HTMLResponse, Response, JSONResponse
from pydantic import BaseModel, Field
from typing import Optional

import core

app = FastAPI(title="Scorecard Wizard")

class GenerateRequest(BaseModel):
    date: str = Field(..., description="YYYY-MM-DD")
    sportId: int
    gameId: int
    fullStats: bool = True
    showLogos: bool = True
    statsDateMode: str = "default"   # default | specify
    statsDate: Optional[str] = None  # YYYY-MM-DD if specify
    filename: str = "scorecard"
    statsStartDate: Optional[str] = None
    statsGameType: Optional[str] = None
    blankIfNoLineups: bool = False

@app.get("/", response_class=HTMLResponse)
def index():
    # Static HTML wizard (kept simple; you can move this into templates later).
    html = open("templates/index.html", "r", encoding="utf-8").read()
    return HTMLResponse(html)

@app.get("/api/sports")
def sports():
    # expose available leagues
    sports = [{"id": k, "name": v["name"]} for k, v in core.SPORTS.items()]
    sports.sort(key=lambda x: x["id"])
    return {"sports": sports}

@app.get("/api/schedule")
def schedule(date: str, sportId: int):
	try:
		return core.list_schedule(date=date, sport_id=sportId)
	except Exception as e:
		raise HTTPException(status_code=400, detail=str(e))



@app.get("/api/manual_preview")
def manual_preview(date: str, sportId: int, gameId: int):
    """
    Returns the 'manual mode' lineup tables as plain text.
    """
    try:
        text = core.manual_preview_text(game_id=gameId, date=date, sport_id=sportId)
        return {"text": text}
    except Exception as e:
        raise HTTPException(status_code=400, detail=str(e))




@app.get("/api/stats_check")
def stats_check(date: str, sportId: int, gameId: int):
    """
    Returns whether stats generation would trigger the original interactive
    'default start date after game' prompt.
    """
    try:
        game, year, month, day, month_str = core._find_game_in_schedule(gameId, date, sportId)
        return core.stats_correction_needed(game=game, sport_id=sportId, date=date)
    except Exception as e:
        raise HTTPException(status_code=400, detail=str(e))


@app.post("/api/generate")
def generate(req: GenerateRequest):
    try:
        pdf_bytes = core.generate_pdf_bytes(
            game_id=req.gameId,
            date=req.date,
            sport_id=req.sportId,
            full_stats=req.fullStats,
            show_logos=req.showLogos,
            stats_date_mode=req.statsDateMode,
            stats_date=req.statsDate,
            stats_start_date=req.statsStartDate,
		    stats_game_type=req.statsGameType,
		    blank_if_no_lineups=req.blankIfNoLineups,
        )
    except Exception as e:
        raise HTTPException(status_code=400, detail=str(e))

    safe_name = "".join(ch for ch in req.filename if ch.isalnum() or ch in ("-", "_", " "))
    safe_name = safe_name.strip().replace(" ", "_") or "scorecard"

    return Response(
        content=pdf_bytes,
        media_type="application/pdf",
        headers={"Content-Disposition": f'inline; filename="{safe_name}.pdf"'},
    )
    
@app.get("/api/stats_defaults")
def stats_defaults(date: str, sportId: int, gameId: int):
    return core.stats_defaults(game_id=gameId, date=date, sport_id=sportId)

